
/* Tool for the creation and extracting of ebook RFF filetypes  */
/* Written by Edmond Franks, (c) 2001                           */
/* This tool is released under the GNU Public License           */
/* This code relies on information from public sources,         */
/* including the reverse engineered specifications published    */
/* into the Internet.                                           */

/* What a RFF is?
/* A Gemstar ebook, model 1100 uses the RFF format for          */
/* executionable files.  Examples of such files include the     */
/* main reader code, named "ram.rff" and unnamed recovery code  */
/* that until the prior hack was downloadable from Gemstar's    */
/* support site.                                                */

/* How looks an RFF file                                        */
/* The RFF file consists of an 512 octet header and a sequence  */
/* of blocks, each 512 bytes in length.  Incomplete blocks must */
/* be padded to the end with the octet containing all ones.     */
/* The format of the header is:                                 */
/* At offset, the data type, means                              */
/* +0   quad octet              DEh C0h 00h A0h is a signature  */
/* +4   sixty octet string      For humans to track the version */
/*                              information.  Isn't read        */
/* +64  twelve octets, data     Parsed version information,     */
/*                              not used by the ebook ROM       */
/* +76  three octets  'PRR'     Describes type of RFF.          */
/*                              only type 'P' is known.         */
/* +79  one octet, FFh          Padding for previous field      */
/* +80  quad octet              Length of complete file         */
/* +84  quad octet              Length of file without header   */
/* +88  quad octet              Where to locate data            */
/*                              Usually 1MB, cannot be used to  */
/*                              overwrite existing software     */
/* +92  quad octet              Always zero, unused             */
/* +96  quad octet              Always zero, unused             */
/* +100 quad octet              First address to execute from   */
/*                              usually 1MB.                    */
/* +104 quad octet              Size of header.                 */
/*                              This is always 512.             */
/* +108 quad octet              Redundant file length field     */
/*                              Unused by ebook ROM.            */
/* +112 one octet               One byte checksum (1)           */
/* +113 one octet,              Always zero, unused             */
/* +114 two octets, FFh         Unused                          */
/* +116 quad octet              Always zero, unused             */
/* +120 quad octet              1 for encryption, 0 unencrypted */
/* +124 quad octet              Uncompressed length,            */
/*                              0 for uncompressed.             */
/* remainder                    FFh, unused                     */

/* Note 1, how checksum is made                                 */
/* Every byte of the file is added together, then the twos-     */
/* complement is used.                                          */

/* Feature unsupported                                          */
/* 1.  Won't write uncompressed rff files.                      */
/* 2.  Won't do encryption/decryption.                          */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "zlib.h"

/* Visual C++ needs O_BINARY type, but POSIX doesn't provide */
#ifndef O_BINARY
#define O_BINARY 0
#endif

#ifdef S_IREAD
#define PERM   S_IREAD|S_IWRITE
#endif
#ifdef _S_IREAD
#define PERM   _S_IREAD|_S_IWRITE
#endif

typedef unsigned char octet;
typedef unsigned long int quartet;

#define RFF_MAGIC 0xa000c0de



quartet rff_header[512 / 4];

const char welcome[] = \"Welcome to rfftool, version 1.00 \n" \
  "Written by Edmund Franks, (c) 2001 \n" \
  "This tool is released under the terms of the GNU Public License.\n\n";

const char usage[] =
  \"To see information about a rff file, just use the filename. \n" \
  "    Example:  rfftool ram.rff \n\n" \
  "To extract a rff file, provide an output filename which doesn't exist\n" \
  "    Example:  rfftool ram.rff newfile.bin \n\n" \
  "To create a new rff file there are many options:\n" \
  "    -v ##.##.###     Specify Version numbers.\n" \
  "    -e <hex number>  Provide an Execution address in Hex.\n" \
  "    -l <hex number>  Change the Location. \n" \
  "    Example:  rfftool -v 1.2.3 newram.rff newfile.bin\n" \
  "       The second filename must already exist.\n\n" \
  "Thank you that is all.\n";

#define BIGBUF_SIZE 65*1024

int
main (int argc, char **argv)
{
  int arg = 1;
  int i;
  char *s;
  char *rff_filename = NULL;
  char *other_filename = NULL;
  int rff_fd, other_fd;

  octet *bigbuf;
  octet *in_buf, *out_buf;

  int readcount, writecount;
  unsigned long int readpos = 0, writepos = 0;

  z_stream zs;
  int zlib_err, zlib_count;

  char *version_string = NULL;
  quartet load_loc = 0xFFFFFFFF, exec_loc = 0xFFFFFFFF;


  fprintf (stderr, welcome);

  bigbuf = (octet *) malloc (BIGBUF_SIZE);
  if (!bigbuf) {
    fprintf (stderr, "HEY!  Unable to get %d bytes of memory for a buffer\n",
	     BIGBUF_SIZE);
    exit (-1);
  }
  in_buf = bigbuf;
  out_buf = bigbuf + (BIGBUF_SIZE / 2);

  while (arg < argc) {
    s = argv[arg++];
    if (!s)
      break;

    if (*s != '-') {
      if (!rff_filename)
	rff_filename = s;
      else if (!other_filename)
	other_filename = s;
      else {
	fprintf (stderr, usage);
	exit (-1);
      }
      continue;
    }
    s++;
    while (*s) {
      switch ((*s) | 0x20) {
      case 'v':
	if (arg < argc)
	  version_string = argv[arg++];
	else {
	  fprintf (stderr, "HEY! You must supply a version string.\n\n");
	  fprintf (stderr, usage);
	  exit (-1);
	}
	break;
      case 'e':
	if ((arg >= argc) || (sscanf (argv[arg++], "%x", &exec_loc) != 1)) {
	  fprintf (stderr, "HEY! You must supply an execution location.\n\n");
	  fprintf (stderr, usage);
	  exit (-1);
	}
	break;
      case 'l':
	if ((arg >= argc) || (sscanf (argv[arg++], "%x", &load_loc) != 1)) {
	  fprintf (stderr, "HEY! You must supply a load location.\n\n");
	  fprintf (stderr, usage);
	  exit (-1);
	}
	break;
      default:
	fprintf (stderr, "HEY.  I do not recognize option '%c'\n\n", *s);
	fprintf (stderr, usage);
	exit (-1);
      }
      s++;
    }
  }
  if (!rff_filename) {
    fprintf (stderr, usage);
    exit (-1);
  }

  zs.zalloc = (alloc_func) NULL;
  zs.zfree = (free_func) NULL;
  zs.opaque = (voidpf) NULL;


  rff_fd = open (rff_filename, O_RDONLY | O_BINARY);
  if (rff_fd != -1) {
    readcount = read (rff_fd, rff_header, sizeof (rff_header));
    if (readcount == -1) {
      fprintf (stderr, "HEY, I cannot read from file %s.\n"
	       "The O/S reports error %d\n", rff_filename, errno);
      exit (-30);
    }
    if (readcount != sizeof (rff_header)) {
      fprintf (stderr, "HEY, your file %s is too small for an rff.\n"
	       "The O/S reports error %d\n", rff_filename, errno);
      exit (-40);

    }
    if (rff_header[0] != RFF_MAGIC) {
      fprintf (stderr, "HEY, your file %s has an odd magic number.\n" \
	       "I wanted to see %08lx, but your file has %08lx instead.\n",
	       rff_filename, RFF_MAGIC, rff_header[0]);
      exit (-50);
    }
    if (!other_filename) {
      octet *vp;

      /* display function.  The file is no longer needed, although    */
      /* future uses might include actually checking the file lengths */
      /* and checksums against the displayed version                  */
      close (rff_fd);

      printf ("Version:           \"%.60s\"\n", &rff_header[1]);
      vp = (octet *) & rff_header[64 / 4];
      printf ("Alt version:       %02x %02x %02x %02x %02x %02x %02x %02x "
	      "%02x %02x %02x %02x \n",
	      vp[0], vp[1], vp[2], vp[3], vp[4], vp[5], vp[6], vp[7],
	      vp[8], vp[9], vp[10], vp[11]);
      printf ("Type string:       \"%.3s\"\n", &rff_header[76 / 4]);
      printf ("File length:       (hex) %08lx  (dec) %d\n",
	      rff_header[80 / 4], rff_header[80 / 4]);
      printf ("Data length:       (hex) %08lx  (dec) %d\n",
	      rff_header[84 / 4], rff_header[84 / 4]);
      printf ("Load location:     (hex) %08lx  (dec) %d\n",
	      rff_header[88 / 4], rff_header[88 / 4]);
      printf ("Execute location:  (hex) %08lx  (dec) %d\n",
	      rff_header[100 / 4], rff_header[100 / 4]);
      printf ("Header size:       (hex) %08lx  (dec) %d\n",
	      rff_header[104 / 4], rff_header[104 / 4]);
      printf ("2nd File length:   (hex) %08lx  (dec) %d\n",
	      rff_header[108 / 4], rff_header[108 / 4]);
      printf ("Checksum:          (hex) %08lx\n", rff_header[112 / 4]);
      printf ("Encryption flag:   (dec) %d\n", rff_header[120 / 4]);
      printf ("Uncompressed size: (hex) %08lx  (dec) %d\n",
	      rff_header[124 / 4], rff_header[124 / 4]);

      exit (0);
    }
    if (rff_header[120 / 4]) {
      fprintf (stderr, "Hey, encrypted files not yet supported! \n");
      exit (-55);
    }
    other_fd =
      open (other_filename, O_WRONLY | O_CREAT | O_BINARY | O_EXCL, PERM);
    if (other_fd == -1) {
      if (errno == EEXIST) {
	fprintf (stderr, "HEY, you already have a file named \"%s\"\n",
		 other_filename);
	exit (-60);
      }
      else {
	fprintf (stderr, "HEY, I can't make a file named \"%s\"\n"
		 "The O/S told me the error code was %d\n",
		 other_filename, errno);
	exit (-70);
      }
    }
    zlib_err = inflateInit (&zs);
    if (zlib_err != Z_OK) {
      fprintf (stderr, "HEY - ZLIB FAILED \"inflateInit\" with code %d\n",
	       zlib_err);
      close (rff_fd);
      close (other_fd);
      (void) unlink (other_filename);
      exit (-80);
    }
    zs.avail_in = 0;
    zs.next_in = NULL;
    readcount = 1;
    readpos = 0;
    do {
      if (readcount && (zs.avail_in == 0)) {
	readcount = read (rff_fd, in_buf, (BIGBUF_SIZE / 2));
	if (readcount < 0) {
	  fprintf (stderr, "Failed to read %s, at about %d\n" \
		   "O/S error number is %d\n", rff_filename, readpos, errno);
	  close (rff_fd);
	  close (other_fd);
	  (void) unlink (other_filename);
	  exit (-90);
	}
	if (readcount > 0) {
	  readpos += readcount;
	  zs.next_in = in_buf;
	  zs.avail_in = (uInt) readcount;
	}
      }
      zs.next_out = out_buf;
      zs.avail_out = (BIGBUF_SIZE / 2);
      zlib_err = inflate (&zs, Z_SYNC_FLUSH);
      zlib_count = (BIGBUF_SIZE / 2) - zs.avail_out;
      if ((zlib_err != Z_STREAM_END) && (zlib_err != Z_OK)) {
	fprintf (stderr, "HEY, Decompression failed at offset %d, "
		 "with ZLIB error %d\n", readpos, zlib_err);
	close (rff_fd);
	close (other_fd);
	(void) unlink (other_filename);
	exit (-100);
      }
      writecount = write (other_fd, out_buf, zlib_count);
      if (writecount != zlib_count) {
	fprintf (stderr, "HEY! Failed to write to file \"%s\" " \
		 " trying to write %d bytes, only wrote %d \n",
		 other_filename, zlib_count, writecount);
	if (writecount == -1) {
	  fprintf (stderr, "O/S reports error %d\n", errno);
	}
	close (rff_fd);
	close (other_fd);
	(void) unlink (other_filename);
	exit (-90);
      }
    } while (zlib_err == Z_OK);
    fprintf (stderr, "SUCCESS - decompressed %ld octets\n", zs.total_out);

    zlib_err = inflateEnd (&zs);
    if (zlib_err != Z_OK) {
      fprintf (stderr, "HEY, Decompression (\"inflateEnd\") failed at "
	       "offset %d, with ZLIB error %d!\n", readpos, zlib_err);
      close (rff_fd);
      close (other_fd);
      (void) unlink (other_filename);
      exit (-110);
    }
    close (rff_fd);
    close (other_fd);
    exit (0);
  }
  else {
    int flush_type;
    quartet checksum;

    readpos = writepos = 0;

    if (!other_filename) {
      fprintf (stderr, "HEY! I can't create a rff file "
	       "without another filename.\n");
      exit (-115);
    }

    other_fd = open (other_filename, O_RDONLY | O_BINARY);
    if (other_fd == -1) {
      fprintf (stderr, "HEY, I can't read the file named \"%s\"\n"
	       "The O/S told me the error code was %d\n",
	       other_filename, errno);
      exit (-117);
    }

    rff_fd =
      open (rff_filename, O_EXCL | O_WRONLY | O_CREAT | O_BINARY, PERM);

    if (rff_fd == -1) {
      fprintf (stderr, "HEY, I can't make a file named \"%s\"\n"
	       "The O/S told me the error code was %d\n",
	       rff_filename, errno);
      exit (-120);
    }

    writecount = write (rff_fd, rff_header, sizeof (rff_header));
    if (writecount != sizeof (rff_header)) {
      fprintf (stderr, "HEY! Failed to write to file \"%s\" " \
	       " trying to write %d bytes, only wrote %d \n",
	       rff_filename, sizeof (rff_header), writecount);
      if (writecount == -1) {
	fprintf (stderr, "O/S reports error %d\n", errno);
      }
      close (rff_fd);
      close (other_fd);
      (void) unlink (rff_filename);
      exit (-125);
    }

    /* Constants are determined by trial and error to result in identical */
    /* to Gemstar supplied rff files.  */
    zlib_err = deflateInit2 (&zs, 6, Z_DEFLATED, 15, 8, Z_DEFAULT_STRATEGY);
    if (zlib_err != Z_OK) {
      fprintf (stderr, "HEY - ZLIB FAILED \"deflateInit2\" with code %d\n",
	       zlib_err);
      close (rff_fd);
      close (other_fd);
      (void) unlink (rff_filename);
      exit (-130);
    }
    zs.avail_in = 0;
    zs.next_in = NULL;
    flush_type = Z_NO_FLUSH;
    checksum = 0;

    readcount = 1;
    do {
      int zlib_count;

      if (readcount && (zs.avail_in == 0)) {
	readcount = read (other_fd, in_buf, (BIGBUF_SIZE / 2));
	if (readcount < 0) {
	  fprintf (stderr, "Failed to read %s, at about %d\n" \
		   "O/S error number is %d\n", other_filename, readpos,
		   errno);
	  close (rff_fd);
	  close (other_fd);
	  (void) unlink (rff_filename);
	  exit (-140);
	}
	else if (readcount > 0) {
	  readpos += readcount;
	  zs.next_in = in_buf;
	  zs.avail_in = (uInt) readcount;
	}
	else
	  flush_type = Z_FINISH;
      }
      zs.next_out = out_buf;
      zs.avail_out = (BIGBUF_SIZE / 2);
      zlib_err = deflate (&zs, flush_type);
      zlib_count = (BIGBUF_SIZE / 2) - zs.avail_out;

      if ((zlib_err != Z_STREAM_END) && (zlib_err != Z_OK)) {
	fprintf (stderr, "HEY, Compression failed at read offset %d, "
		 " write offset %d with ZLIB error %d\n",
		 readpos, writepos, zlib_err);
	close (rff_fd);
	close (other_fd);
	(void) unlink (rff_filename);
	exit (-150);
      }
      for (i = 0; i < zlib_count; i++)
	checksum += out_buf[i];

      writecount = write (rff_fd, out_buf, zlib_count);
      if (writecount != zlib_count) {
	fprintf (stderr, "HEY! Failed to write to file \"%s\" " \
		 " trying to write %d bytes, only wrote %d \n",
		 rff_filename, zlib_count, writecount);
	if (writecount == -1) {
	  fprintf (stderr, "O/S reports error %d\n", errno);
	}
	close (rff_fd);
	close (other_fd);
	(void) unlink (rff_filename);
	exit (-160);
      }
      writepos += zlib_count;

    } while (zlib_err == Z_OK);
    if ((writepos % 512) != 0) {
      int padding_count;

      padding_count = 512 - (writepos % 512);
      memset (out_buf, 0xFF, padding_count);
      for (i = 0; i < padding_count; i++)
	checksum += out_buf[i];
      writecount = write (rff_fd, out_buf, padding_count);
      writepos += padding_count;
      if (writecount != padding_count) {
	fprintf (stderr, "HEY! Failed to write to file \"%s\" " \
		 " trying to write %d bytes, only wrote %d \n",
		 rff_filename, padding_count, writecount);
	if (writecount == -1) {
	  fprintf (stderr, "O/S reports error %d\n", errno);
	}
	close (rff_fd);
	close (other_fd);
	(void) unlink (rff_filename);
	exit (-170);
      }
    }
    fprintf (stderr, "SUCCESS - Compressed %ld -> %ld octets\n", readpos,
	     writepos);

    zlib_err = deflateEnd (&zs);
    if (zlib_err != Z_OK) {
      fprintf (stderr, "HEY, Compress (\"deflateEnd\") failed at "
	       "offset %d, with ZLIB error %d\n", readpos, zlib_err);
      close (rff_fd);
      close (other_fd);
      (void) unlink (rff_filename);
      exit (-180);
    }

    lseek (rff_fd, 0, SEEK_SET);
    memset (rff_header + (128 / 4), 0xFF, sizeof (rff_header) - 128);
    memset (rff_header, 0x0, 128);
    rff_header[0] = RFF_MAGIC;
    if (exec_loc == 0xFFFFFFFF)
      exec_loc = 1024 * 1024;
    if (load_loc == 0xFFFFFFFF)
      load_loc = 1024 * 1024;
    if (version_string) {
      char *p;

      s = (char *) &rff_header[1];
      sprintf (s, "Ram File %.60s", version_string);

      p = (char *) &rff_header[64 / 4];

      /* These values appear to be fixed.  They aren't currently checked. */
      p[0] = 0x1a;
      p[1] = 0x4;
      p[2] = 0x2;

      /* Start with 3 characters from the last version string */
      s = strrchr (version_string, '.');
      if (s) {
	*s = '\0';
	strncpy (&p[3], s + 1, 3);
	s = strrchr (version_string, '.');
      }
      if (s) {
	*s = '\0';
	strncpy (&p[9], s + 1, 3);
	strncpy (&p[6], version_string, 3);
      }

    }
    rff_header[76 / 4] = 0xFF525250;	/* PRR */
    rff_header[80 / 4] = writepos + 512;
    rff_header[84 / 4] = writepos;
    rff_header[88 / 4] = load_loc;
    rff_header[100 / 4] = exec_loc;
    rff_header[104 / 4] = sizeof (rff_header);
    rff_header[108 / 4] = writepos + 512;
    rff_header[112 / 4] = 0xFFFF0000 | ((-checksum) & 0xFF);
    /* Decryption not supported */
    rff_header[120 / 4] = 0;
    rff_header[124 / 4] = readpos;

    writecount = write (rff_fd, rff_header, sizeof (rff_header));
    if (writecount != sizeof (rff_header)) {
      fprintf (stderr, "HEY! Failed to write to file \"%s\" " \
	       " trying to write %d bytes, only wrote %d \n",
	       rff_filename, sizeof (rff_header), writecount);
      if (writecount == -1) {
	fprintf (stderr, "O/S reports error %d\n", errno);
      }
      close (rff_fd);
      close (other_fd);
      (void) unlink (rff_filename);
      exit (-190);
    }
    close (rff_fd);
    close (other_fd);
    exit (0);
  }
}
